#ifndef __IWindow__
#define __IWindow__

#include <MathTools/CPoint.hpp>
#include <MathTools/CRect.hpp>
#include <MathTools/CDimension.hpp>
#include <Basics/CString.hpp>
#include "../Listeners/IFocusListener.hpp"
#include "../Listeners/IWindowChangeListener.hpp"
#include "../Listeners/IMouseListener.hpp"
#include "../Listeners/IDropFileListener.hpp"
#include "../Listeners/IKeyboardListener.hpp"
#include "../Graphics/CGraphics.hpp"
#include "../Basics/ITimedObject.hpp"
#include "SWindowHandle.hpp"
#include "CWindowAttributes.hpp"

//	===========================================================================

using Exponent::MathTools::CPoint;
using Exponent::MathTools::CRect;
using Exponent::Basics::CString;
using Exponent::GUI::Listeners::IDropFileListener;
using Exponent::GUI::Listeners::IFocusListener;
using Exponent::GUI::Listeners::IKeyboardListener;
using Exponent::GUI::Listeners::IMouseListener;
using Exponent::GUI::Listeners::IWindowChangeListener;
using Exponent::GUI::Graphics::CGraphics;
using Exponent::GUI::Basics::ITimedObject;
using Exponent::GUI::Windowing::SWindowHandle;
using Exponent::GUI::Windowing::CWindowAttributes;

//	===========================================================================

/**
 * @cond
 */
#ifndef __ICONTROLROOT_GUARD__
#define __ICONTROLROOT_GUARD__
	namespace Exponent { namespace GUI { namespace Controls { interface IControlRoot; } } }
	using Exponent::GUI::Controls::IControlRoot;
#endif
/**
 * @endcond
 */

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Windowing
		{
			/**
			 * @interface IWindow IWindow.hpp
			 * @brief Interface to a window. All window classes must implement this interface
			 *
			 * @date 18/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: IWindow.hpp,v 1.6 2007/02/11 02:16:22 paul Exp $
			 */
			interface IWindow : public ICountedObject
			{

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * Construction
				 */
				IWindow() { }

				/**
				 * Destruction
				 */
				virtual ~IWindow() { }

//	===========================================================================

				/**
				 * Set the parent window of this window
				 * @param parent The parent window
				 * @note For future use
				 */
				virtual void setParentWindow(IWindow *parent) = 0;

				/**
				 * Get hte parent window
				 * @retval IWindow* The parent window, may be NULL
				 * @note For future use
				 */
				virtual IWindow *getParentWindow() const = 0;

//	===========================================================================

				/**
				 * Register a child window
				 * @param child The child window
				 * @note For future use
				 */
				virtual void registerChildWindow(IWindow *child) = 0;

				/**
				 * Unregister all child windows
				 * @note For future use
				 */
				virtual void unregisterAllChildWindows() = 0;

				/**
				 * Unregister a specific child window
				 * @param child The child window to unregister
				 * @note For future use
				 */
				virtual void unregisterChldWindow(IWindow *child) = 0;

//	===========================================================================

				/**
				 * Get the widow handle
				 * @retval const SWindowHandle* The handle of this window
				 */
				virtual const SWindowHandle *getWindowHandle() const = 0;

				/**
				 * Get the widow handle
				 * @retval SWindowHandle* The handle of this window
				 */
				virtual SWindowHandle *getMutableWindowHandle() const = 0;

				/**
				 * Get the widow attributes
				 * @retval CWindowAttributes* The attributes of this window
				 */
				virtual CWindowAttributes *getWindowAttributes() const = 0;

//	===========================================================================

				/**
				 * Initilalise the window
				 * @param windowAttributes The window attributes. (copy should be made, do not store this pointer - it may be deleted...)
				 * @param showWindow Should the window be shown immediately
				 */
				virtual void initialiseWindow(CWindowAttributes *windowAttributes, const bool showWindow) = 0;

				/**
				 * Uninitialise the window
				 */
				virtual void uninitialiseWindow() = 0;

//	===========================================================================

				/**
				 * Open the window
				 */
				virtual void openWindow() = 0;

				/**
				 * Close the widnow
				 */
				virtual void closeWindow() = 0;

				/**
				 * Destroy the widnow
				 * @note clears internal buffers, and uninitialises the window
				 */
				virtual void destroyWindow() = 0;

				/**
				 * Update the window redraws entire window
				 */
				virtual void updateWindow() = 0;

				/**
				 * Disable the window
				 */
				virtual void disableWindow() = 0;

				/**
				 * Enable the window
				 */
				virtual void enableWindow() = 0;

//	===========================================================================

				/**
				 * Draw the window images...
				 * @param graphics The graphics handle to use
				 */
				virtual void draw(CGraphics &graphics) = 0;

				/**
				 * Set an area of the window dirty
				 * @param area The area of the window relative to its top left to draw
				 */
				virtual void redrawWindow(const CRect &area) = 0;

				/**
				 * Set the whole window dirty - generally use sparingly!!
				 */
				virtual void redrawWindow() = 0;

//	===========================================================================

				/**
				 * Set the position of the window relative to its parent
				 * @param position The position relative to the parent, usually the desktop
				 */
				virtual void setWindowPosition(const CPoint &position) = 0;

				/**
				 * Get the postion relative to the parent
				 * @retval const CPoint& The window position relative to the parent, usually the desktop
				 */
				virtual const CPoint &getWindowPosition() const = 0;

				/**
				 * Set the size of the widnwo
				 * @param dimension The dimension of the window
				 */
				virtual void setWindowSize(const CDimension &dimension) = 0;

				/**
				 * Get the size of the window
				 * @retval const CDimension& The dimension of the window
				 */
				virtual const CDimension &getWindowSize() const = 0;

				/**
				 * Set the size and position
				 * @param positionAndSize The new position and size of the window
				 */
				virtual void setWindowPositionAndSize(const CRect &positionAndSize) = 0;

				/**
				 * Get the size and position
				 * @retval const CRect& The size and position of the window
				 * @see getWindowSize
				 * @see getWindowPosition
				 */
				virtual const CRect &getWindowPositionAndSize() const = 0;

				/**
				 * Set the window alpha value
				 * @param alpha The alpha level 0 - 1
				 */
				virtual void setWindowAlpha(const double alpha) = 0;

//	===========================================================================

				/**
				 * Handle a timer event happening - Intended for internal use
				 * @param id The id of the timer that has fired an event
				 */
				virtual void handleTimerEvent(const long id) = 0;

				/**
				 * Start a timer
				 * @param id The unique id of the timer object to start
				 * @param timeInMilliseconds The time in milliseconds between updates
				 */
				virtual void startTimer(const long id, const long timeInMilliseconds) = 0;

				/**
				 * Stop a timer
				 * @param id The unique id of the timer
				 */
				virtual void stopTimer(const long id) = 0;

				/**
				 * Add a new timed object
				 * @param timedObject The object to add to the timer collection
				 */
				virtual void addTimedObject(ITimedObject *timedObject) = 0;

				/**
				 * Remove a timed objects
				 * @param timedObject The object to remove from timing
				 */
				virtual void removeTimedObject(ITimedObject *timedObject) = 0;

				/**
				 * Clear the timed bojects
				 */
				virtual void clearTimedObjects() = 0;

//	===========================================================================

				/**
				 * Register a focus listener
				 * @param listener The listener for window focus information
				 */
				virtual void registerFocusListener(IFocusListener *listener) = 0;

				/**
				 * Register a window change listener
				 * @param listener The listener for window change information
				 */
				virtual void registerWindowChangeListener(IWindowChangeListener *listener) = 0;

				/**
				 * Register a mouse listener
				 * @param listener The listener for mouse information
				 */
				virtual void registerMouseListener(IMouseListener *listener) = 0;

				/**
				 * Register a drop file listener
				 * @param listener The listener for drop file information
				 */
				virtual void registerDropFileListener(IDropFileListener *listener) = 0;

				/**
				 * Register keyboard listener
				 * @param listener The listener for keyboard information
				 */
				virtual void registerKeyboardListener(IKeyboardListener *listener) = 0;

//	===========================================================================

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle mouse events, false otherwise
				 */
				virtual bool isMouseEnabled() const = 0;

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle keyboard events, false otherwise
				 */
				virtual bool isKeyboardEnabled() const = 0;

				/**
				 * Can the window handle dropped files??
				 * @retval bool True if window can handle dropped files, false otherwise
				 */
				virtual bool isDropFileEnabled() const = 0;

//	===========================================================================

				/**
				 * Get the mouse
				 * @retval CMouse* The mouse for this window
				 */
				virtual CMouse *getMouse() = 0;

				/**
				 * Get the control root
				 * @retval IControlRoot* The control root of this window
				 */
				virtual IControlRoot *getControlRoot() const = 0;

//	===========================================================================

			};
		}
	}
}
#endif	// End of IWindow.hpp